/*
 * MyApp_SensorM.nc
 */
 
includes sensorboardApp;

/**
 * This module shows how to use the Timer, LED, ADC and Messaging components.
 * Sensor messages are broadcast single-hop over the RF radio
 *
 * @author Crossbow Technology Inc.
 **/


//#define ADDR TOS_LOCAL_ADDRESS?TOS_BCAST_ADDR:TOS_UART_ADDR
#define ADDR TOS_BCAST_ADDR

module MyAppM {
  provides {
    interface StdControl;
  }
  uses {
    interface Timer;
    interface Leds;
	interface SendMsg;
	interface ReceiveMsg;

	//Battery
    interface StdControl as BattControl;
    interface ADC as ADCBATT;
	//Photo
	interface StdControl as PhotoControl;
	interface ADC as Light;
	//Temp
	interface StdControl as TempControl;
	interface ADC as Temperature;
	//Mic
	interface StdControl as MicControl;
	interface Mic;
	interface ADC as MicADC;
  }
}
implementation {
//#define DBG_PKT 1
//#define SO_DEBUG 1
//#include <sodebug.h>

  norace bool sending_packet;
  norace bool sensing;
  TOS_Msg msg_buffer;
  XDataMsg *pack;

 

/*--Task--*/
  task void SendPacket ();
  


  static void start() 
    {
      call BattControl.start();
      call MicControl.start();
      call TempControl.start();  
      call PhotoControl.start(); 
    }

  /**
   * Initialize the component.
   * 
   * @return Always returns <code>SUCCESS</code>
   **/
  command result_t StdControl.init() {
    call BattControl.init();
    call Leds.init(); 
    call TempControl.init();
	call PhotoControl.init();
    call MicControl.init();
    call Mic.muxSel(1);  // Set the mux so that raw microhpone output is selected
    call Mic.gainAdjust(64);  // Set the gain of the microphone. (refer to Mic)
 
    // Initialize the message packet with default values
    atomic {
	  sending_packet = FALSE;
	  sensing = FALSE;

      pack = (XDataMsg *)&(msg_buffer.data);
      pack->xSensorHeader.board_id = SENSOR_BOARD_ID;
      pack->xSensorHeader.packet_id = 3;
	  pack->xSensorHeader.measurement = FALSE;
	  pack->xSensorHeader.priority = 0;
      pack->idHeader.node_id = TOS_LOCAL_ADDRESS;
	  pack->idHeader.dest_id = TOS_LOCAL_ADDRESS;
	  pack->idHeader.seqno = 0;
	  pack->idHeader.app_id = 5;

	}
    return SUCCESS;
  }

  /**
   * Start things up.  This just sets the rate for the clock component.
   * 
   * @return Always returns <code>SUCCESS</code>
   **/
  command result_t StdControl.start() {
    // Start a repeating timer that fires every 1000ms
  
    return call Timer.start(TIMER_REPEAT, 1000);
  }

  /**
   * Halt execution of the application.
   * This just disables the clock component.
   * 
   * @return Always returns <code>SUCCESS</code>
   **/
  command result_t StdControl.stop() {

	return call Timer.stop();

  }

  /**
   * Toggle the red LED in response to the <code>Timer.fired</code> event.  
   * Start the Light sensor control and sample the data
   *
   * @return Always returns <code>SUCCESS</code>
   **/
  event result_t Timer.fired()
  {
	call Leds.redToggle();
    if (!sensing){
		start();
		call ADCBATT.getData();
		atomic sensing = TRUE;
    }

	return SUCCESS;
  }

  

   
     task void battstop()
    {
  	call BattControl.stop();
    }
    task void tempstop()
    {
  	call TempControl.stop();
    }
    task void photostop()
    {
  	call PhotoControl.stop();
    }
    task void Micstop()
    {
  	call MicControl.stop();
    }
	
  /***************************************************************************
 * Battery Ref  or thermistor data ready
 ***************************************************************************/
  async event result_t ADCBATT.dataReady(uint16_t data) {
      if (!sensing) return FAIL;
      pack->xSensorData.vref = data;
      atomic sensing = FALSE;
      post battstop();
      call Temperature.getData();
      return SUCCESS;
  }
 
    
  /***************************************************************************
 * Temperature ADC data ready 
 * Read and get next channel.
 **************************************************************************/ 
  async event result_t Temperature.dataReady(uint16_t data) {
      pack->xSensorData.thermistor = data;
      post tempstop(); 
	  call Light.getData(); 
      return SUCCESS;
  }
 /***************************************************************************
 * Photocell ADC data ready 
 * Read and get next channel.
 **************************************************************************/ 
  async event result_t Light.dataReady(uint16_t data) {
	  atomic	pack->xSensorData.light = data;
      post photostop();
      call MicADC.getData();
      return SUCCESS;
  }
/***************************************************************************
 * MicroPhone ADC data ready
 * Read and toggle sounder.
 * send uart packet
 **************************************************************************/
  async event result_t MicADC.dataReady(uint16_t data) {
     atomic pack->xSensorData.mic = data;//data;
     post Micstop();
	 post SendPacket();
	 return SUCCESS;
  }
 /**
   * Sensor data message has been sucessfully sent over the radio
   * Toggle green LED to signal message sent
   *
   * @return Always returns <code>SUCCESS</code>
   **/ 
  event result_t SendMsg.sendDone(TOS_MsgPtr msg, result_t success) {
    call Leds.greenToggle();
    atomic sending_packet = FALSE;
	
    return SUCCESS;
  }
  
  event TOS_MsgPtr ReceiveMsg.receive(TOS_MsgPtr m){

	return m;
  }
  

  task void SendPacket () {
	if (!sending_packet){
		atomic sending_packet = TRUE;

		// Send message over radio
		if (call SendMsg.send(ADDR,sizeof(XDataMsg),&msg_buffer) != SUCCESS) sending_packet = FALSE;
	}
	return;
  }
}


